package gov.va.vamf.scheduling.direct.domain;

import agilex.commons.NullChecker;
import com.agilex.healthcare.veteranappointment.domain.Namespace;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import java.util.Collection;

@XmlAccessorType(value = XmlAccessType.NONE)
@XmlRootElement(namespace = Namespace.DirectScheduling)
public class StopCode {

    @XmlElement
    String primary;

    @XmlElement
    String secondary;

    public StopCode() {}

    public StopCode(String primary, String secondary) {
        this.primary = primary;
        this.secondary = secondary;
    }

    public String getPrimary() {
        return primary;
    }

    public void setPrimary(String primary) {
        this.primary = primary;
    }

    public String getSecondary() {
        return secondary;
    }

    public void setSecondary(String secondary) {
        this.secondary = secondary;
    }

    /**
     * Determine if two stop codes match. Two stop codes match if corresponding primary and secondary codes are
     * equal; null primary or secondary codes in the "base" stop code are always considered a match against the
     * corresponding value.
     *
     * E.g.:
     * StopCode.match(new StopCode('123', '456'), new StopCode('123', '456'))); // true
     * StopCode.match(new StopCode(null, '456'), new StopCode('123', '456')));  // true
     * StopCode.match(new StopCode('123', null), new StopCode('123', '456')));  // true
     * StopCode.match(new StopCode('111', '222'), new StopCode('123', '456'))); // false
     *
     * @param base
     * @param code
     * @return
     */
    public static boolean match(StopCode base, StopCode code) {
        boolean result = false;
        final String basePrimary = base.getPrimary();
        final String baseSecondary = base.getSecondary();

        if(NullChecker.isNullish(basePrimary) || basePrimary.equals(code.getPrimary())) {
            if(NullChecker.isNullish(baseSecondary)) {
                result = true;
            } else {
                result = baseSecondary.equals(code.getSecondary());
            }
        } else {
            result = false;
        }
        return result;
    }

    /**
     * Determine if a stop code matches any of a collection of stop codes.
     *
     * @param matchAgainstStopCodes
     * @param stopCode
     * @return
     * @see #match(StopCode, StopCode)
     */
    public static boolean matchAny(final Collection<StopCode> matchAgainstStopCodes, final StopCode stopCode) {
        for (final StopCode matchAgainstStopCode : matchAgainstStopCodes) {
            if (StopCode.match(matchAgainstStopCode, stopCode)) {
                return true;
            }
        }
        return false;
    }
}
